/*
 * Copyright 2012-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.autoconfigure.jms.activemq;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.DeprecatedConfigurationProperty;

/**
 * Configuration properties for ActiveMQ.
 *
 * @author Greg Turnquist
 * @author Stephane Nicoll
 * @author Aurélien Leboulanger
 * @author Venil Noronha
 */
@ConfigurationProperties(prefix = "spring.activemq")
public class ActiveMQProperties {

	/**
	 * URL of the ActiveMQ broker. Auto-generated by default.
	 */
	private String brokerUrl;

	/**
	 * Whether the default broker URL should be in memory. Ignored if an explicit broker
	 * has been specified.
	 */
	private boolean inMemory = true;

	/**
	 * Login user of the broker.
	 */
	private String user;

	/**
	 * Login password of the broker.
	 */
	private String password;

	/**
	 * Time to wait before considering a close complete.
	 */
	private Duration closeTimeout = Duration.ofSeconds(15);

	/**
	 * Whether to stop message delivery before re-delivering messages from a rolled back
	 * transaction. This implies that message order is not preserved when this is enabled.
	 */
	private boolean nonBlockingRedelivery = false;

	/**
	 * Time to wait on message sends for a response. Set it to 0 to wait forever.
	 */
	private Duration sendTimeout = Duration.ofMillis(0);

	private Pool pool = new Pool();

	private Packages packages = new Packages();

	public String getBrokerUrl() {
		return this.brokerUrl;
	}

	public void setBrokerUrl(String brokerUrl) {
		this.brokerUrl = brokerUrl;
	}

	public boolean isInMemory() {
		return this.inMemory;
	}

	public void setInMemory(boolean inMemory) {
		this.inMemory = inMemory;
	}

	public String getUser() {
		return this.user;
	}

	public void setUser(String user) {
		this.user = user;
	}

	public String getPassword() {
		return this.password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public Duration getCloseTimeout() {
		return this.closeTimeout;
	}

	public void setCloseTimeout(Duration closeTimeout) {
		this.closeTimeout = closeTimeout;
	}

	public boolean isNonBlockingRedelivery() {
		return this.nonBlockingRedelivery;
	}

	public void setNonBlockingRedelivery(boolean nonBlockingRedelivery) {
		this.nonBlockingRedelivery = nonBlockingRedelivery;
	}

	public Duration getSendTimeout() {
		return this.sendTimeout;
	}

	public void setSendTimeout(Duration sendTimeout) {
		this.sendTimeout = sendTimeout;
	}

	public Pool getPool() {
		return this.pool;
	}

	public void setPool(Pool pool) {
		this.pool = pool;
	}

	public Packages getPackages() {
		return this.packages;
	}

	public static class Pool {

		/**
		 * Whether a PooledConnectionFactory should be created, instead of a regular
		 * ConnectionFactory.
		 */
		private boolean enabled;

		/**
		 * Whether to block when a connection is requested and the pool is full. Set it to
		 * false to throw a "JMSException" instead.
		 */
		private boolean blockIfFull = true;

		/**
		 * Blocking period before throwing an exception if the pool is still full.
		 */
		private Duration blockIfFullTimeout = Duration.ofMillis(-1);

		/**
		 * Whether to create a connection on startup. Can be used to warm up the pool on
		 * startup.
		 */
		private boolean createConnectionOnStartup = true;

		/**
		 * Connection expiration timeout.
		 */
		private Duration expiryTimeout = Duration.ofMillis(0);

		/**
		 * Connection idle timeout.
		 */
		private Duration idleTimeout = Duration.ofSeconds(30);

		/**
		 * Maximum number of pooled connections.
		 */
		private int maxConnections = 1;

		/**
		 * Maximum number of active sessions per connection.
		 */
		private int maximumActiveSessionPerConnection = 500;

		/**
		 * Reset the connection when a "JMSException" occurs.
		 */
		private boolean reconnectOnException = true;

		/**
		 * Time to sleep between runs of the idle connection eviction thread. When
		 * negative, no idle connection eviction thread runs.
		 */
		private Duration timeBetweenExpirationCheck = Duration.ofMillis(-1);

		/**
		 * Whether to use only one anonymous "MessageProducer" instance. Set it to false
		 * to create one "MessageProducer" every time one is required.
		 */
		private boolean useAnonymousProducers = true;

		public boolean isEnabled() {
			return this.enabled;
		}

		public void setEnabled(boolean enabled) {
			this.enabled = enabled;
		}

		public boolean isBlockIfFull() {
			return this.blockIfFull;
		}

		public void setBlockIfFull(boolean blockIfFull) {
			this.blockIfFull = blockIfFull;
		}

		public Duration getBlockIfFullTimeout() {
			return this.blockIfFullTimeout;
		}

		public void setBlockIfFullTimeout(Duration blockIfFullTimeout) {
			this.blockIfFullTimeout = blockIfFullTimeout;
		}

		@DeprecatedConfigurationProperty
		public boolean isCreateConnectionOnStartup() {
			return this.createConnectionOnStartup;
		}

		public void setCreateConnectionOnStartup(boolean createConnectionOnStartup) {
			this.createConnectionOnStartup = createConnectionOnStartup;
		}

		@DeprecatedConfigurationProperty(reason = "Use idle-timeout instead")
		public Duration getExpiryTimeout() {
			return this.expiryTimeout;
		}

		public void setExpiryTimeout(Duration expiryTimeout) {
			this.expiryTimeout = expiryTimeout;
		}

		public Duration getIdleTimeout() {
			return this.idleTimeout;
		}

		public void setIdleTimeout(Duration idleTimeout) {
			this.idleTimeout = idleTimeout;
		}

		public int getMaxConnections() {
			return this.maxConnections;
		}

		public void setMaxConnections(int maxConnections) {
			this.maxConnections = maxConnections;
		}

		public int getMaximumActiveSessionPerConnection() {
			return this.maximumActiveSessionPerConnection;
		}

		public void setMaximumActiveSessionPerConnection(
				int maximumActiveSessionPerConnection) {
			this.maximumActiveSessionPerConnection = maximumActiveSessionPerConnection;
		}

		@DeprecatedConfigurationProperty(
				reason = "Disabling this option will likely lead to broken connections in the pool.")
		public boolean isReconnectOnException() {
			return this.reconnectOnException;
		}

		public void setReconnectOnException(boolean reconnectOnException) {
			this.reconnectOnException = reconnectOnException;
		}

		public Duration getTimeBetweenExpirationCheck() {
			return this.timeBetweenExpirationCheck;
		}

		public void setTimeBetweenExpirationCheck(Duration timeBetweenExpirationCheck) {
			this.timeBetweenExpirationCheck = timeBetweenExpirationCheck;
		}

		public boolean isUseAnonymousProducers() {
			return this.useAnonymousProducers;
		}

		public void setUseAnonymousProducers(boolean useAnonymousProducers) {
			this.useAnonymousProducers = useAnonymousProducers;
		}

	}

	public static class Packages {

		/**
		 * Whether to trust all packages.
		 */
		private Boolean trustAll;

		/**
		 * Comma-separated list of specific packages to trust (when not trusting all
		 * packages).
		 */
		private List<String> trusted = new ArrayList<>();

		public Boolean getTrustAll() {
			return this.trustAll;
		}

		public void setTrustAll(Boolean trustAll) {
			this.trustAll = trustAll;
		}

		public List<String> getTrusted() {
			return this.trusted;
		}

		public void setTrusted(List<String> trusted) {
			this.trusted = trusted;
		}

	}

}
