/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.api.internal.cache;

import java.io.File;
import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.CheckForNull;
import org.sonarsource.scanner.api.internal.cache.FileHashes;
import org.sonarsource.scanner.api.internal.cache.Logger;

public class FileCache {
    private final Path dir;
    private final Path tmpDir;
    private final FileHashes hashes;
    private final Logger logger;

    FileCache(Path dir, FileHashes fileHashes, Logger logger) {
        this.hashes = fileHashes;
        this.logger = logger;
        this.dir = this.createDir(dir, "user cache: ");
        logger.info(String.format("User cache: %s", dir.toString()));
        this.tmpDir = this.createDir(dir.resolve("_tmp"), "temp dir");
    }

    static FileCache create(Path dir, Logger logger) {
        return new FileCache(dir, new FileHashes(), logger);
    }

    public File getDir() {
        return this.dir.toFile();
    }

    @CheckForNull
    public File get(String filename, String hash) {
        Path cachedFile = this.dir.resolve(hash).resolve(filename);
        if (Files.exists(cachedFile, new LinkOption[0])) {
            return cachedFile.toFile();
        }
        this.logger.debug(String.format("No file found in the cache with name %s and hash %s", filename, hash));
        return null;
    }

    public File get(String filename, String hash, Downloader downloader) {
        Path hashDir = this.hashDir(hash);
        Path targetFile = hashDir.resolve(filename);
        if (!Files.exists(targetFile, new LinkOption[0])) {
            Path tempFile = this.newTempFile();
            FileCache.download(downloader, filename, tempFile);
            String downloadedHash = this.hashes.of(tempFile.toFile());
            if (!hash.equals(downloadedHash)) {
                throw new IllegalStateException("INVALID HASH: File " + tempFile.toAbsolutePath() + " was expected to have hash " + hash + " but was downloaded with hash " + downloadedHash);
            }
            FileCache.mkdirQuietly(hashDir);
            this.renameQuietly(tempFile, targetFile);
        }
        return targetFile.toFile();
    }

    private static void download(Downloader downloader, String filename, Path tempFile) {
        try {
            downloader.download(filename, tempFile.toFile());
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to download " + filename + " to " + tempFile, e);
        }
    }

    private void renameQuietly(Path sourceFile, Path targetFile) {
        try {
            Files.move(sourceFile, targetFile, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException ex) {
            this.logger.warn(String.format("Unable to rename %s to %s", sourceFile.toAbsolutePath(), targetFile.toAbsolutePath()));
            this.logger.warn("A copy/delete will be tempted but with no guarantee of atomicity");
            try {
                Files.move(sourceFile, targetFile, new CopyOption[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to move " + sourceFile.toAbsolutePath() + " to " + targetFile, e);
            }
        }
        catch (FileAlreadyExistsException ex) {
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to move " + sourceFile.toAbsolutePath() + " to " + targetFile, e);
        }
    }

    private Path hashDir(String hash) {
        return this.dir.resolve(hash);
    }

    private static void mkdirQuietly(Path hashDir) {
        try {
            Files.createDirectories(hashDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create cache directory: " + hashDir, e);
        }
    }

    private Path newTempFile() {
        try {
            return Files.createTempFile(this.tmpDir, "fileCache", null, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp file in " + this.tmpDir, e);
        }
    }

    private Path createDir(Path dir, String debugTitle) {
        this.logger.debug("Create: " + dir.toString());
        try {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to create " + debugTitle + dir.toString(), e);
        }
        return dir;
    }

    @FunctionalInterface
    public static interface Downloader {
        public void download(String var1, File var2) throws IOException;
    }
}

