/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.fabric.gateway.impl;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import javax.json.Json;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.json.JsonString;
import javax.json.JsonWriter;
import org.hyperledger.fabric.gateway.impl.GatewayUtils;
import org.hyperledger.fabric.gateway.spi.Checkpointer;

public final class FileCheckpointer
implements Checkpointer {
    private static final Set<OpenOption> OPEN_OPTIONS = Collections.unmodifiableSet(EnumSet.of(StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE));
    private static final int VERSION = 1;
    private static final String CONFIG_KEY_VERSION = "version";
    private static final String CONFIG_KEY_BLOCK = "block";
    private static final String CONFIG_KEY_TRANSACTIONS = "transactions";
    private final Path filePath;
    private final FileChannel fileChannel;
    private final Reader fileReader;
    private final Writer fileWriter;
    private final AtomicLong blockNumber = new AtomicLong(-1L);
    private final Set<String> transactionIds = Collections.newSetFromMap(new ConcurrentHashMap());

    public FileCheckpointer(Path checkpointFile) throws IOException {
        boolean isFileAlreadyPresent = Files.exists(checkpointFile, new LinkOption[0]);
        this.filePath = checkpointFile;
        this.fileChannel = FileChannel.open(this.filePath, OPEN_OPTIONS, new FileAttribute[0]);
        this.lockFile();
        CharsetDecoder utf8Decoder = StandardCharsets.UTF_8.newDecoder();
        this.fileReader = Channels.newReader(this.fileChannel, utf8Decoder, -1);
        CharsetEncoder utf8Encoder = StandardCharsets.UTF_8.newEncoder();
        this.fileWriter = Channels.newWriter(this.fileChannel, utf8Encoder, -1);
        if (isFileAlreadyPresent) {
            this.load();
        } else {
            this.save();
        }
    }

    private void lockFile() throws IOException {
        FileLock fileLock;
        try {
            fileLock = this.fileChannel.tryLock();
        }
        catch (OverlappingFileLockException e) {
            throw new IOException("File is already locked: " + this.filePath, e);
        }
        if (fileLock == null) {
            throw new IOException("Another process holds an overlapping lock for file: " + this.filePath);
        }
    }

    private synchronized void load() throws IOException {
        this.fileChannel.position(0L);
        JsonObject savedData = this.loadJson();
        int version = savedData.getInt(CONFIG_KEY_VERSION, 0);
        if (version != 1) {
            throw new IOException("Unsupported checkpoint data version " + version + " from file: " + this.filePath);
        }
        this.parseDataV1(savedData);
    }

    private JsonObject loadJson() throws IOException {
        JsonReader jsonReader = Json.createReader((Reader)this.fileReader);
        try {
            return jsonReader.readObject();
        }
        catch (RuntimeException e) {
            throw new IOException("Failed to parse checkpoint data from file: " + this.filePath, e);
        }
    }

    private void parseDataV1(JsonObject json) throws IOException {
        try {
            this.blockNumber.set(json.getJsonNumber(CONFIG_KEY_BLOCK).longValue());
            this.transactionIds.clear();
            json.getJsonArray(CONFIG_KEY_TRANSACTIONS).getValuesAs(JsonString.class).stream().map(JsonString::getString).forEach(this.transactionIds::add);
        }
        catch (RuntimeException e) {
            throw new IOException("Bad format of checkpoint data from file: " + this.filePath, e);
        }
    }

    private synchronized void save() throws IOException {
        JsonObject jsonData = this.buildJson();
        this.fileChannel.position(0L);
        this.saveJson(jsonData);
        this.fileChannel.truncate(this.fileChannel.position());
    }

    private JsonObject buildJson() {
        return Json.createObjectBuilder().add(CONFIG_KEY_VERSION, 1).add(CONFIG_KEY_BLOCK, this.blockNumber.get()).add(CONFIG_KEY_TRANSACTIONS, Json.createArrayBuilder(this.transactionIds)).build();
    }

    private void saveJson(JsonObject json) throws IOException {
        JsonWriter jsonWriter = Json.createWriter((Writer)this.fileWriter);
        try {
            jsonWriter.writeObject(json);
        }
        catch (RuntimeException e) {
            throw new IOException("Failed to write checkpoint data to file: " + this.filePath, e);
        }
        this.fileWriter.flush();
    }

    @Override
    public long getBlockNumber() {
        return this.blockNumber.get();
    }

    @Override
    public synchronized void setBlockNumber(long blockNumber) throws IOException {
        this.blockNumber.set(blockNumber);
        this.transactionIds.clear();
        this.save();
    }

    @Override
    public Set<String> getTransactionIds() {
        return Collections.unmodifiableSet(this.transactionIds);
    }

    @Override
    public synchronized void addTransactionId(String transactionId) throws IOException {
        this.transactionIds.add(transactionId);
        this.save();
    }

    @Override
    public void close() throws IOException {
        this.fileChannel.close();
    }

    public String toString() {
        return GatewayUtils.toString(this, "file=" + this.filePath, "blockNumber=" + this.blockNumber.get(), "transactionIds=" + this.transactionIds);
    }
}

