package cn.axj.crypt.secret.implement;

import cn.axj.crypt.constant.AlgorithmConstant;
import cn.axj.crypt.secret.BaseAlgorithmHttpInputMessage;
import cn.axj.crypt.secret.config.RsaKeyConfig;
import org.apache.commons.codec.binary.Base64;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * @author aoxiaojun
 * @date 2020/12/11 15:30
 **/
@Component
public class RsaAlgorithmHttpInputMessage extends BaseAlgorithmHttpInputMessage {

    @Resource
    private RsaKeyConfig rsaKeyConfig;


    @Override
    public String getBinding() {
        return AlgorithmConstant.RSA;
    }

    @Override
    public String decrypt(String content) throws Exception {
        return decode(content);
    }

    @Override
    public String encrypt(String content) throws Exception {
        return this.encrypt(content,getPublicKey());
    }


    private static final String KEY_ALGORTHM = "RSA";
    private static final String SIGNATURE_ALGORITHM = "SHA256withRSA";
    private static final String DEFAULT_CHARSET = "UTF-8";

    private static String publicKeyStr;
    private static String privateKeyStr;

    @Autowired
    public void setPublicKeyStr() {
        RsaAlgorithmHttpInputMessage.publicKeyStr = rsaKeyConfig.getPublicKey();
    }

    @Autowired
    public void setPrivateKeyStr() {
        RsaAlgorithmHttpInputMessage.privateKeyStr = rsaKeyConfig.getPrivateKey();
    }

    /**
     * 分段加密
     * @param content
     * @param publicKey
     * @return
     * @throws Exception
     */
    private String encrypt(String content, PublicKey publicKey) throws Exception {
        Cipher cipher = Cipher.getInstance("rsa");
        cipher.init(Cipher.ENCRYPT_MODE, publicKey);
        Base64 base64 = new Base64();
        byte[] data = content.getBytes();
        int inputLen = data.length;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offSet = 0;
        for(int i = 0; inputLen - offSet > 0; offSet = i * 117) {
            byte[] cache;
            if(inputLen - offSet > 117) {
                cache = cipher.doFinal(data, offSet, 117);
            } else {
                cache = cipher.doFinal(data, offSet, inputLen - offSet);
            }
            out.write(cache, 0, cache.length);
            ++i;
        }
        byte[] encryptedData = out.toByteArray();
        out.close();
        return base64.encodeToString(encryptedData);
    }

    /**
     * 分段解密
     * @param content
     * @param privateKey
     * @return
     * @throws Exception
     */
    private String decrypt(String content, PrivateKey privateKey) throws Exception {
        byte[] byteArray = Base64.decodeBase64(content);
        Cipher cipher = Cipher.getInstance(privateKey.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, privateKey);

        int inputLen = byteArray.length;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offSet = 0;
        for(int i = 0; inputLen - offSet > 0; offSet = i * 128) {
            byte[] cache;
            if(inputLen - offSet > 128) {
                cache = cipher.doFinal(byteArray, offSet, 128);
            } else {
                cache = cipher.doFinal(byteArray, offSet, inputLen - offSet);
            }
            out.write(cache, 0, cache.length);
            ++i;
        }
        byte[] decryptedData = out.toByteArray();
        out.close();
        return new String(decryptedData);

    }

    private PrivateKey getPrivateKey() throws Exception {
        byte[] keyBytes = decryptBase64(RsaAlgorithmHttpInputMessage.privateKeyStr);
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORTHM);
        return keyFactory.generatePrivate(pkcs8EncodedKeySpec);
    }

    private PublicKey getPublicKey() throws Exception {
        byte[] keyBytes = decryptBase64(RsaAlgorithmHttpInputMessage.publicKeyStr);
        //构造X509EncodedKeySpec对象
        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(keyBytes);
        //指定加密算法
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORTHM);
        return keyFactory.generatePublic(x509EncodedKeySpec);
    }

    private byte[] decryptBase64(String keys) {
        Base64 base64 = new Base64();
        return base64.decode(keys);
    }

    public  String encode(String content) throws Exception {
        return encrypt(content, getPublicKey());
    }

    public String decode(String content) throws Exception {
        return decrypt(content, getPrivateKey());
    }


    /**
     * 测试
     */
    public static void main(String[] args) throws Exception {
        RsaAlgorithmHttpInputMessage message = new RsaAlgorithmHttpInputMessage();
        String content = "{\"userId\":\"234567890\"}";
        System.out.println("加密前：" + content);
         String publicKey = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDJqAEi0zbr23xGAHFPuWQLCWf4LlDduXsXgGiJ+xjhU+kbJ4xCZ6ovovGkHj5IWT2NwfSIN1bB+VjEG55CrQ/+5az+0aF9pdDEgB62NtBbrYaICtVTSA79nujqF3aoJLi/x4L/rhkbDHulvr85BLkAUFwYC+C/isZP1tF4Ha9kyQIDAQAB";

         String privateKey = "MIICeAIBADANBgkqhkiG9w0BAQEFAASCAmIwggJeAgEAAoGBAMmoASLTNuvbfEYAcU+5ZAsJZ/guUN25exeAaIn7GOFT6RsnjEJnqi+i8aQePkhZPY3B9Ig3VsH5WMQbnkKtD/7lrP7RoX2l0MSAHrY20FuthogK1VNIDv2e6OoXdqgkuL/Hgv+uGRsMe6W+vzkEuQBQXBgL4L+Kxk/W0Xgdr2TJAgMBAAECgYEAtnZIiN06LZEMUj6mSG/D4OSi6YHm86iXgv1SGb9UlNWa71IKbC9mYvH52PjwfAAbXP/pJbOE0cCvG8qtmUEEXZD+hCFKmm9uix6GTX8PsFq6+byjxgaoyEiMiCJrfwvwxuQIPRjYK93r1cRrxFG6oYt9cHSdeJKfpeocy4GjsXECQQDlAwmkHWmQASPGA1xLDY84FEwfLEMLvCEAiUCOydjzEZnGXVismjc+HTqez0mhdxounHohi4eDiNijWfRrFda1AkEA4Wuv8N7SJCpe+ihsYRp1wsgLC0AytLLGcLQt+Ey4toEDCoO/V0wFzYzDkFkYuebkqL+ujaHLe5+85b0XLlguRQJAWHbH7qF9+aa3DvKVYFiojl0o2pHahFnz1aUvTvkh2SNfZ4T/kA0yccgQsPAZ3UmIz6CafQLNIj5mkRNam1MgUQJBALEke+2QXJj5BwwUqh6guXvo0ez0dSRU/OZiYWGngNxCg85GfB2w6zp9bM1JhsnILVaC0zwPsiUjAUGUl9vehfUCQQCHh/SDQqE8kvVbMSIwl4a3N6eHQzOwJpjQMo8DuJh6u+2+LYLHGaoZ3keBLyLbNNyGElfBSwmBpeMLW6FTAK0T";

        System.out.println("加密public：" + publicKey);
        String encrypt = message.encrypt(content,message.getTestPublicKey(publicKey));
        System.out.println("加密后：" + encrypt);
//        String user_id = new DESString().decrypt("3ak+WSRGZzTNp6LT2zfShA==");
        String decrypt = message.decrypt(encrypt, message.getTestPrivateKey(privateKey));
        System.out.println("解密后：" + decrypt);
    }

    private PrivateKey getTestPrivateKey(String privateKeyStr) throws Exception {
        byte[] keyBytes = decryptBase64(privateKeyStr);
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORTHM);
        return keyFactory.generatePrivate(pkcs8EncodedKeySpec);
    }

    private PublicKey getTestPublicKey(String publicKeyStr) throws Exception {
        byte[] keyBytes = decryptBase64(publicKeyStr);
        //构造X509EncodedKeySpec对象
        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(keyBytes);
        //指定加密算法
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORTHM);
        return keyFactory.generatePublic(x509EncodedKeySpec);
    }

}
