package cn.axj.crypt.secret.implement;

import cn.axj.crypt.constant.AlgorithmConstant;
import cn.axj.crypt.secret.BaseAlgorithmHttpInputMessage;
import cn.axj.crypt.secret.config.AesEcbConfig;
import org.apache.commons.codec.binary.Base64;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import sun.misc.BASE64Decoder;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.SecretKeySpec;

/**
 * Description: 自定义AES解密 不使用偏移量
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved.
 *
 * @version 1.0  2022/01/2022/1/19 6:01 PM by 张振涛（zhangzt@cloud-young.com）创建
 */
@Component
public class AesEcbAlgorithmHttpInputMessage extends BaseAlgorithmHttpInputMessage {

    @Resource
    private AesEcbConfig aesKeyConfig;
    /**
     * aes的key
     */
    private static String aesKey;

    /**
     * 算法名
     */
    private static final String KEY_ALGORITHM = "AES";

    /**
     * 加解密算法/模式/填充方式
     * ECB模式只用密钥即可对数据进行加密解密，CBC模式需要添加一个参数iv
     */
    private static final String CIPHER_ALGORITHM = "AES/ECB/PKCS5Padding";

    @Autowired
    public void setAesKey() {
        AesEcbAlgorithmHttpInputMessage.aesKey = aesKeyConfig.getKey();
    }

    @Override
    public String getBinding() {
        return AlgorithmConstant.AES_ECB;
    }

    /**
     * Description: 解密
     * @version 1.0  2022/1/19 9:40 PM by 张振涛（zhentao.zhang1@newtouch.com）创建
     * @param content
     * @return
     * @throws Exception
     */
    @Override
    public String decrypt(String content) throws Exception {
        return aesDecrypt(content, aesKey);
    }

    /**
     * Description: 加密
     * @version 1.0  2022/1/19 9:41 PM by 张振涛（zhentao.zhang1@newtouch.com）创建
     * @param content
     * @return
     * @throws Exception
     */
    @Override
    public String encrypt(String content) throws Exception {
        return aesEncrypt(content, aesKey);
    }

    /**
     * AES加密为base 64 code
     * @param content 待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的base 64 code
     * @throws Exception
     */
    public static String aesEncrypt(String content, String encryptKey) throws Exception {
        return base64Encode(aesEncryptToBytes(content, encryptKey));
    }

    /**
     * base 64 encode
     * @param bytes 待编码的byte[]
     * @return 编码后的base 64 code
     */
    public static String base64Encode(byte[] bytes){
        return Base64.encodeBase64String(bytes);
    }

    /**
     * AES加密
     * @param content 待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的byte[]
     * @throws Exception
     */
    public static byte[] aesEncryptToBytes(String content, String encryptKey) throws Exception {
        KeyGenerator kgen = KeyGenerator.getInstance(KEY_ALGORITHM);
        kgen.init(128);
        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        cipher.init(Cipher.ENCRYPT_MODE, new SecretKeySpec(encryptKey.getBytes(), KEY_ALGORITHM));

        return cipher.doFinal(content.getBytes("utf-8"));
    }

    /**
     * 将base 64 code AES解密
     * @param encryptStr 待解密的base 64 code
     * @param decryptKey 解密密钥
     * @return 解密后的string
     * @throws Exception
     */
    public static String aesDecrypt(String encryptStr, String decryptKey) throws Exception {
        return encryptStr.isEmpty() ? null : aesDecryptByBytes(base64Decode(encryptStr), decryptKey);
    }

    /**
     * AES解密
     * @param encryptBytes 待解密的byte[]
     * @param decryptKey 解密密钥
     * @return 解密后的String
     * @throws Exception
     */
    public static String aesDecryptByBytes(byte[] encryptBytes, String decryptKey) throws Exception {
        KeyGenerator kgen = KeyGenerator.getInstance("AES");
        kgen.init(128);

        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(decryptKey.getBytes(), "AES"));
        byte[] decryptBytes = cipher.doFinal(encryptBytes);
        return new String(decryptBytes);
    }

    /**
     * base 64 decode
     * @param base64Code 待解码的base 64 code
     * @return 解码后的byte[]
     * @throws Exception
     */
    public static byte[] base64Decode(String base64Code) throws Exception{
        return base64Code.isEmpty() ? null : new BASE64Decoder().decodeBuffer(base64Code);
    }

    /**
     * 测试
     */
    public static void main(String[] args) throws Exception {
        String content = "{\"userId\":\"234567890\"}";
        System.out.println("加密前：" + content);
        System.out.println("加密密钥和解密密钥：" + "abcdefgabcdefg12");
        String encrypt = aesEncrypt(content,"abcdefgabcdefg12");
        System.out.println("加密后：" + encrypt);
//        String user_id = new DESString().decrypt("3ak+WSRGZzTNp6LT2zfShA==");
        String decrypt = aesDecrypt(encrypt,"abcdefgabcdefg12");
        System.out.println("解密后：" + decrypt);
    }
}
